import ast
import glob
import socket
from urllib import parse
from pprint import pprint

import ldap


def connect_local():
    ldapi_path = glob.glob('/run/slapd-*.socket')[0]
    ldap_conn = ldap.initialize(
        'ldapi://%s' % (
            parse.quote_plus(ldapi_path)
        )
    )
    ldap_conn.sasl_non_interactive_bind_s('EXTERNAL')
    return ldap_conn


def get_main_dn():
    fqdn = socket.getfqdn()
    main_dn = 'dc=' + ',dc='.join(fqdn.split('.')[1:])
    return main_dn


def search_in_ldap(connect, base_dn, search_filter):
    ldap_search_scope = ldap.SCOPE_SUBTREE
    attrs = ['envvar']
    elems = connect.search_s(base_dn, ldap_search_scope, search_filter, attrs)
    return elems


def process_ldap_data(data):
    result = []
    for record in data:
        if 'envvar' in record[1]:
            decoded_list = [ast.literal_eval(item.decode()) for item in record[1]['envvar']]
            merged_dict = {key: val for k in decoded_list for key, val in k.items()}
            result.append(merged_dict)
    for dict_item in result:
        for k, v in dict_item.items():
            if v == 'None':
                dict_item[k] = None
    return result


def generate_random_string():
    randomize_cmd = "openssl rand -hex 30"
    try:
        result = __salt__["cmd.run"](randomize_cmd)
    except NameError:
        import subprocess
        import shlex
        result = subprocess.check_output(shlex.split(randomize_cmd)).decode().rstrip()
    return result


def generate_all_passwords():
    result = {
        'core': generate_random_string(),
        'async': generate_random_string(),
        'helpcenter': generate_random_string(),
        'syncer': generate_random_string(),
        'salt': generate_random_string(),
        'zabbbix': generate_random_string(),
        'common_env': generate_random_string(),
        'adcan': generate_random_string(),
        'role': generate_random_string(),
        'salt_db': generate_random_string()
    }
    return result


def get_services_passwords(connect, main_dn):
    base_dn = f'cn=services,cn=aldpro,cn=etc,{main_dn}'
    search_filter = '(&(cn=*)(objectclass=rbta-aldpro-service-instance))'
    data = search_in_ldap(connect, base_dn, search_filter)
    subsystems_vars_decoded_list = process_ldap_data(data)
    result = {}
    for vars_dict in subsystems_vars_decoded_list:
        if 'POSTGRES_DB' in vars_dict and vars_dict['POSTGRES_DB'] == 'osinstall':
            result['os'] = vars_dict['POSTGRES_PASSWORD']
        elif 'DB_NAME' in vars_dict and vars_dict['DB_NAME'] == 'repo':
            result['repo'] = vars_dict['DB_PASSWORD']
    if result.get('os') is None:
        result['os'] = generate_random_string()
    if result.get('repo') is None:
        result['repo'] = generate_random_string()
    return result


def get_master_passwords(connect, main_dn):
    base_dn = f'cn=masters,cn=ipa,cn=etc,{main_dn}'
    search_filter = '(&(cn=*)(objectclass=rbta-aldpro-service-instance))'
    data = search_in_ldap(connect, base_dn, search_filter)
    vars_master = process_ldap_data(data)[0]
    result = {
        'core': vars_master.get('CORE_DB_PASSWORD', generate_random_string()),
        'async': vars_master.get('ASYNC_DB_PASSWORD', generate_random_string()),
        'helpcenter': vars_master.get('HELPCENTER_DB_PASSWORD', generate_random_string()),
        'syncer': vars_master.get('SYNCER_DB_PASSWORD', generate_random_string()),
        'salt': vars_master.get('SS_PASSWORD', generate_random_string()),
        'zabbbix': vars_master.get('ZABBIX_PSW', generate_random_string()),
        'common_env': vars_master.get('ENV_PASSWORD', generate_random_string()),
        'adcan': vars_master.get('RABBIT_PASSWORD', generate_random_string()),
        'role': vars_master.get('ROLESERVICE_PASSWORD', generate_random_string()),
        'salt_db': vars_master.get('ADCAN_DB_PASSWORD', generate_random_string())
    }
    return result


def run():
    result = {}
    try:
        connect = connect_local()
        main_dn = get_main_dn()
        master_passwords = get_master_passwords(connect, main_dn)
        services_passwords = get_services_passwords(connect, main_dn)
        connect.unbind_s()
        result.update(master_passwords)
        result.update(services_passwords)
        return result
    except:
        result = generate_all_passwords()
        return result


if __name__ == '__main__':
    pprint(run())
